﻿using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Forms;
using WPF.PresentationLayer.Resources;
using WPF.PresentationLayer.UC115.Events;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    /// <summary>
    /// Blood Unit Tabs view model. Contains the Abo/Rh and Antigen Typing tabs
    /// </summary>
    public class BloodUnitTabsViewModel : ViewModelBase
    {
        #region Properties
        private readonly IList<IPendingBloodUnitTestModel> _pendingBloodTestList;

        private int selectedTabIndex;
        /// <summary>
        /// SelectedTabIndex
        /// </summary>
        public int SelectedTabIndex
        {
            get
            {
                return selectedTabIndex;
            }
            set
            {
                selectedTabIndex = value;
                RaisePropertyChanged(() => SelectedTabIndex);
            }
        }

        private TabItem _selectedTab;
        public TabItem SelectedTab
        {
            get
            {
                return _selectedTab;
            }
            set
            {
                _selectedTab = value;

                // The tab view's data does not get set immediately. If we wait
                // until the UI is idle then we can set the help
                System.Windows.Application.Current.Dispatcher.BeginInvoke(new System.Action(() =>
                {
                    if (SelectedTab != null)
                    {
                        var view = SelectedTab.Content as FrameworkElement;
                        var vm = view == null ? null : view.DataContext as PendingTestViewModelBase;
                        if (vm != null && VbecsBaseForm != null)
                        {
                            VbecsBaseForm.SetThisHelpFile(VbecsBaseForm, vm.HelpFileTopic);
                        }
                    }
                }));
            }
        }

        private BloodUnitTabViewModel bloodUnitsAboRhViewModel;
        /// <summary>
        /// BloodUnitsAboRhViewModel
        /// </summary>
        public BloodUnitTabViewModel BloodUnitsAboRhViewModel
        {
            get
            {
                return bloodUnitsAboRhViewModel;
            }
            set
            {
                bloodUnitsAboRhViewModel = value;
                RaisePropertyChanged(() => BloodUnitsAboRhViewModel);
            }
        }

        private BloodUnitTabViewModel bloodUnitAntigenTypingViewModel;
        /// <summary>
        /// BloodUnitAntigenTypingViewModel
        /// </summary>
        public BloodUnitTabViewModel BloodUnitAntigenTypingViewModel
        {
            get
            {
                return bloodUnitAntigenTypingViewModel;
            }
            set
            {
                bloodUnitAntigenTypingViewModel = value;
                RaisePropertyChanged(() => BloodUnitAntigenTypingViewModel);
            }
        }
        #endregion

        /// <summary>
        ///  BloodUnitTabsViewModel - Constructor
        /// </summary>
        public BloodUnitTabsViewModel()
        {
            LogoHeaderText = "Automated Testing - Pending Blood Units";
            FormText = "VBECS - " + LogoHeaderText;
            SelectedTabIndex = 0;

            PropertyChanged += BloodUnitTabViewModel_PropertyChanged;

            Messenger.Default.Register<PendingTestReviewedEvent>(this, OnPendingTestReviewedEvent);

            _pendingBloodTestList = PendingBloodUnitTest.GetPendingBloodUnitTests(LogonUser.LogonUserDivisionCode.Trim(), PendingTestStatus.Pending);

            this.LockFunction = LockFunctions.UC115AutomatedTestingReview;            
        }

        /// <summary>
        /// Clean up events after the form closed
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void VbecsBaseForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            if (VbecsBaseForm != null)
            {
                Messenger.Default.Unregister(this);
                PropertyChanged -= BloodUnitTabViewModel_PropertyChanged;
                VbecsBaseForm.OnPreviewClosing -= PreviewClosing;
                VbecsBaseForm.FormClosed -= VbecsBaseForm_FormClosed;
            }
        }

        /// <summary>
        /// Fired after a pending test is reviewed (accepted/rejected) in the system
        /// </summary>
        /// <param name="payload"></param>
        private void OnPendingTestReviewedEvent(PendingTestReviewedEvent payload)
        {
            AutoSelectTabWithResults();
        }

        private void AutoSelectTabWithResults()
        {
            if (SelectedTabIndex == 0
                && !BloodUnitsAboRhViewModel.PendingBloodUnitFullList.Any()
                && BloodUnitAntigenTypingViewModel.PendingBloodUnitFullList.Any())
            {
                // No more UnitABO test so auto select the Antigen Typing tab
                SelectedTabIndex = 1;
            }
            else if (SelectedTabIndex == 1
                && BloodUnitsAboRhViewModel.PendingBloodUnitFullList.Any()
                && !BloodUnitAntigenTypingViewModel.PendingBloodUnitFullList.Any())
            {
                // No more Antigen Typing tests so auto select the UnitABO tab
                SelectedTabIndex = 0;
            }

            if (!BloodUnitsAboRhViewModel.PendingBloodUnitFullList.Any() &&
                !BloodUnitAntigenTypingViewModel.PendingBloodUnitFullList.Any())
            {
                // To tests left to review so close the window
                VbecsBaseForm.Close();
            }
        }

        /// <summary>
        /// After the form is set we can construct the two tab view models and populate their data
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void BloodUnitTabViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "VbecsBaseForm" &&
                VbecsBaseForm != null)
            {
                VbecsBaseForm.OnPreviewClosing += PreviewClosing;
                VbecsBaseForm.FormClosed += VbecsBaseForm_FormClosed;                

                if (BloodUnitsAboRhViewModel == null)
                {
                    var aboRhTests = _pendingBloodTestList.Select(s => s as PendingBloodUnitAboRhModel)
                                                          .Where(w => w != null)
                                                          .Select(x => new BloodUnitItemViewModel(x));

                    BloodUnitsAboRhViewModel = new BloodUnitTabViewModel(VbecsBaseForm, 
                                                                         aboRhTests, 
                                                                         StrRes.HelpFiles.UC115.AutomatedTestingBloodUnitAboRhTab());
                }

                if (BloodUnitAntigenTypingViewModel == null)
                {
                    var antigenTypingTests = _pendingBloodTestList.Select(s => s as PendingBloodUnitAntigenTypingModel)
                                       .Where(w => w != null)
                                       .Select(x => new BloodUnitItemViewModel(x));

                    BloodUnitAntigenTypingViewModel = new BloodUnitTabViewModel(VbecsBaseForm, 
                                                                                antigenTypingTests, 
                                                                                StrRes.HelpFiles.UC115.AutomatedTestingBloodUnitAntigentTypingTab());
                }

                AutoSelectTabWithResults();
            }
        }

        private void PreviewClosing(object sender, CancelEventArgs cancelEventArgs)
        {
            // BR_115.15, BR_2.18
            VbecsBaseForm.IsDirty = IsDirty();
        }

        /// <summary>
        /// Check for unsaved changes
        /// </summary>
        /// <returns></returns>
        private bool IsDirty()
        {
            return BloodUnitsAboRhViewModel.IsDirty() ||
                   BloodUnitAntigenTypingViewModel.IsDirty();
        }
    }
}
